/******************************************************************************
 * (C) Copyright 2000 by Agilent Technologies GmbH. All rights reserved.      *
 ******************************************************************************/

/* ---------------------------------------------------------------
 * File: xlatrig.c 
 * trigger sequencer functions                                  
 * -----------------------------------------------------------------*/

#include <xpciapi.h>
#include <xaddrmap.h>

/* Pointer to trigger sequencer database */
#define DB_TRIGSEQ (bx_handlearray[handle].db->Analyzer.TrigSeq[0])

/* Value of generic property */
#define GENERIC(prop) (DB_TRIGSEQ.GenProp[prop])

/* Value of tran property */
#define TRAN(tran,prop) (DB_TRIGSEQ.Line[tran].Transient[prop])

/* Value of cond property */
#define COND(tran,prop) (DB_TRIGSEQ.Line[tran].Condition[prop])


/****************************************************************************
  Generic properties ********************************************************
****************************************************************************/

/*---------------------------------------------------------------------------*
 * BestXTrigGenSet
 *
 * Purpose: Sets a generic property for the trigger sequencer
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXTrigGenSet(
  bx_handletype handle,
  bx_triggentype prop,
  bx_int32 val
)
{
  BX_DECLARE_FUNCNAME ("BestXTrigGenSet");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    if( !BestXCapabilityCheck(handle,BX_CAPABILITY_ANALYZER) ||  /* AMR */
        !BestXCapabilityCheck(handle,BX_CAPABILITY_OBSERVER) )   /* OMR */
    {
      BX_TRY_RET=BX_E_OK;
    }
    else 
    {
      BX_TRY_FAIL(BX_E_NO_CAPABILITY);
    }  

    /* range checking */
    BX_TRY(BestXParamCheck(handle,BX_PARAM_TRIGSEQ_GEN,(bx_int32)prop,val));
    GENERIC(prop)=val;
  }
  BX_ERRETURN(BX_TRY_RET);
}


/*---------------------------------------------------------------------------*
 * BestXTrigGenGet
 *
 * Purpose: Gets a generic property of the trigger sequencer
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXTrigGenGet(bx_handletype handle,
  bx_triggentype prop,
  bx_int32 * val)
{
  BX_DECLARE_FUNCNAME ("BestXTrigGenGet");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    /* 
       No analyzer licensecheck here,
       because DBInit() must always succeed
       (as long as CAPI license available).
    */
    BX_LICENSECHECK(BX_CAPABILITY_CAPI);
   
    /* range checking */
    BX_TRY(BestXParamProbe(handle,BX_PARAM_TRIGSEQ_GEN,(bx_int32)prop));
    BX_TRY_FCT_PARAM_NULL_POINTER(val);
    
    *val=GENERIC(prop);
  }
  BX_ERRETURN(BX_TRY_RET);
}


/*---------------------------------------------------------------------------*
 * BestXTrigGenDefaultSet()
 *
 * Purpose: Sets all generic props for the trigger sequencer to default
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXTrigGenDefaultSet(
  bx_handletype handle
)
{
  BX_DECLARE_FUNCNAME("BestXTrigGenDefaultSet []");

  BX_TRY_VARS_NO_PROG;
  bx_int32 i;

  /* Generic info for range checking */
  const bx_generic_infotype *GenInfo;
  const bx_param_infotype *ParamInfo;

  BX_TRY_BEGIN
  {
    
    if( !BestXCapabilityCheck(handle,BX_CAPABILITY_ANALYZER) ||  /* AMR */
        !BestXCapabilityCheck(handle,BX_CAPABILITY_OBSERVER) )   /* OMR */
    {
      BX_TRY_RET=BX_E_OK;
    }
    else 
    {
      BX_TRY_FAIL(BX_E_NO_CAPABILITY);
    }  
   
    /* Get pointer to generic info */
    BX_TRY(BestXGenInfoGet(handle, BX_PARAM_TRIGSEQ_GEN, &GenInfo));

    for (i = 0; i < GenInfo->num_elem; i++)
    {
      /* Get pointer to i-th (existing) property */
      BX_TRY(BestXParamInfoGet(handle, BX_PARAM_TRIGSEQ_GEN,
          i, &ParamInfo, (bx_int32)BX_INDEX_SEARCH));

      /* Set it to default */
      BX_TRY(BestXTrigGenSet(handle,
            ParamInfo->proptyp.trigseqgenprop,
            ParamInfo->defaultval));
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}

/****************************************************************************
  TRAN properties ********************************************************
****************************************************************************/

/*---------------------------------------------------------------------------*
 * BestXTrigTranSet
 *
 * Purpose: Sets a tran property for the trigger sequencer
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXTrigTranSet(
  bx_handletype handle,
  bx_int32 transient,
  bx_trigtrantype prop,
  bx_int32 val
)
{
  BX_DECLARE_FUNCNAME ("BestXTrigTranSet");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    if( !BestXCapabilityCheck(handle,BX_CAPABILITY_ANALYZER) ||  /* AMR */
        !BestXCapabilityCheck(handle,BX_CAPABILITY_OBSERVER) )   /* OMR */
    {
      BX_TRY_RET=BX_E_OK;
    }
    else 
    {
      BX_TRY_FAIL(BX_E_NO_CAPABILITY);
    }
    /* range checking */
    BX_FCT_PARAM_CHK(1,transient>=BX_TRIGLINE_SIZE); 
    BX_TRY(BestXParamCheck(handle,BX_PARAM_TRIGSEQ_TRAN,(bx_int32)prop,val));
    TRAN(transient,prop)=val;
  
    /* Track the number of lines with maximum-function.
       Caution: 
       transient is zero-based, MaxNumLines is one-based ! 
    */
    if (transient+1 > DB_TRIGSEQ.MaxNumLines)
    {
      DB_TRIGSEQ.MaxNumLines=transient+1;
    }
  }
  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXTrigTranGet
 *
 * Purpose: Gets a tran property for the trigger sequencer
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXTrigTranGet(
  bx_handletype handle,
  bx_int32 transient,
  bx_trigtrantype prop,
  bx_int32 *val
)
{
  BX_DECLARE_FUNCNAME ("BestXTrigTranGet");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    /* range checking */
    BX_FCT_PARAM_CHK(1,transient>=BX_TRIGLINE_SIZE); 
    BX_TRY(BestXParamProbe(handle,BX_PARAM_TRIGSEQ_TRAN,(bx_int32)prop));
    BX_TRY_FCT_PARAM_NULL_POINTER(val);

    *val=TRAN(transient,prop);
  }
  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXTrigTranDefaultSet()
 *
 * Purpose: Sets all tran props for a transient to default
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXTrigTranDefaultSet(
  bx_handletype handle,
  bx_int32 transient
)
{
  BX_DECLARE_FUNCNAME("BestXTrigTranDefaultSet []");

  BX_TRY_VARS_NO_PROG;
  bx_int32 i;

  /* Generic info for range checking */
  const bx_generic_infotype *GenInfo;
  const bx_param_infotype *ParamInfo;

  BX_TRY_BEGIN
  {
    /* license checking */

    /* 
       No analyzer licensecheck here,
       because DBInit() must always succeed
       (as long as CAPI license available).
    */
    BX_LICENSECHECK(BX_CAPABILITY_CAPI);

    BX_FCT_PARAM_CHK(1,transient>=BX_TRIGLINE_SIZE); 

    /* Get pointer to generic info */
    BX_TRY(BestXGenInfoGet(handle, BX_PARAM_TRIGSEQ_TRAN, &GenInfo));

    for (i = 0; i < GenInfo->num_elem; i++)
    {
      /* Get pointer to i-th (existing) property */
      BX_TRY(BestXParamInfoGet(handle, BX_PARAM_TRIGSEQ_TRAN,
          i, &ParamInfo, (bx_int32)BX_INDEX_SEARCH));

      /* Set it to default */
      BX_TRY(BestXTrigTranSet(handle,transient,
            ParamInfo->proptyp.trigseqtranprop,
            ParamInfo->defaultval));
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}

/****************************************************************************
  COND properties ********************************************************
****************************************************************************/

/*---------------------------------------------------------------------------*
 * BestXTrigCondSet
 *
 * Purpose: Sets a cond property for trigger sequencer
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXTrigCondSet(
  bx_handletype handle,
  bx_int32 transient,
  bx_trigcondtype prop,
  bx_charptrtype cond
)
{
  BX_DECLARE_FUNCNAME ("BestXTrigCondSet");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {

    if( !BestXCapabilityCheck(handle,BX_CAPABILITY_ANALYZER) ||  /* AMR */
        !BestXCapabilityCheck(handle,BX_CAPABILITY_OBSERVER) )   /* OMR */
    {
      BX_TRY_RET=BX_E_OK;
    }
    else 
    {
      BX_TRY_FAIL(BX_E_NO_CAPABILITY);
    }

    /* range checking */
    BX_FCT_PARAM_CHK(1,transient>=BX_TRIGLINE_SIZE); 
    BX_TRY(BestXParamProbe(handle,BX_PARAM_TRIGSEQ_COND,(bx_int32)prop));
    BX_TRY_FCT_PARAM_NULL_POINTER(cond);
    
    if (COND(transient,prop))
    {
      /* first free the old string */
      BestXMemFree((void **) &(COND(transient,prop))); 
      COND(transient,prop)=NULL;
    }
    
    /* Allocate the new string */
    COND(transient,prop)=(bx_charptrtype) BestXMemCalloc(BESTX_STRLEN(cond)+1,sizeof(bx_int8));
    BX_TRY_FAIL((COND(transient,prop))?BX_E_OK:BX_E_HOST_MEM_FULL);
    
    /* Set property value */
    BESTX_STRCPY(COND(transient,prop),cond);

    /* Track the number of lines with maximum-function.
       Caution: 
       transient is zero-based, MaxNumLines is one-based ! 
    */
    if (transient+1 > DB_TRIGSEQ.MaxNumLines)
    {
      DB_TRIGSEQ.MaxNumLines=transient+1;
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXTrigCondGet
 *
 * Purpose: Gets a cond property for the trigger sequencer
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXTrigCondGet(
  bx_handletype handle,
  bx_int32 transient,
  bx_trigcondtype prop,
  bx_charptrtype *cond
)
{
  BX_DECLARE_FUNCNAME ("BestXTrigCondGet");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    /* range checking */
    BX_FCT_PARAM_CHK(1,transient>=BX_TRIGLINE_SIZE); 
    BX_TRY(BestXParamProbe(handle,BX_PARAM_TRIGSEQ_COND,(bx_int32)prop));
    BX_TRY_FCT_PARAM_NULL_POINTER(cond);

    *cond=COND(transient,prop);
  }
  BX_ERRETURN(BX_TRY_RET);
}


/*---------------------------------------------------------------------------*
 * BestXTrigCondDefaultSet()
 *
 * Purpose: Sets all cond props for a transient to default
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXTrigCondDefaultSet(
  bx_handletype handle,
  bx_int32 transient
)
{
  BX_DECLARE_FUNCNAME("BestXTrigCondDefaultSet []");

  BX_TRY_VARS_NO_PROG;
  bx_int32 i;

  /* Generic info for range checking */
  const bx_generic_infotype *GenInfo;
  const bx_param_infotype *ParamInfo;

  BX_TRY_BEGIN
  {
    /* license checking */
    /* 
       No analyzer licensecheck here,
       because DBInit() must always succeed
       (as long as CAPI license available).
    */
    BX_LICENSECHECK(BX_CAPABILITY_CAPI);

    BX_FCT_PARAM_CHK(1,transient>=BX_TRIGLINE_SIZE); 

    /* Get pointer to generic info */
    BX_TRY(BestXGenInfoGet(handle, BX_PARAM_TRIGSEQ_COND, &GenInfo));

    for (i = 0; i < GenInfo->num_elem; i++)
    {
      /* Get pointer to i-th (existing) property */
      BX_TRY(BestXParamInfoGet(handle, BX_PARAM_TRIGSEQ_COND,
          i, &ParamInfo, (bx_int32)BX_INDEX_SEARCH));

      /* Set it to default */
      BX_TRY(BestXTrigCondSet(handle,transient,
            ParamInfo->proptyp.trigseqcondprop,
            ParamInfo->defaultstr));
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}

/****************************************************************************
  TRAN + COND properties ****************************************************
****************************************************************************/

/*---------------------------------------------------------------------------*
 * BestXTrigTranCondDefaultSet()
 *
 * Purpose: Sets all trans and cond props for a transient to default
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXTrigTranCondDefaultSet(
  bx_handletype handle,
  bx_int32 transient         
)
{
  BX_DECLARE_FUNCNAME("BestXTrigTranCondDefaultSet []");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {

    if( !BestXCapabilityCheck(handle,BX_CAPABILITY_ANALYZER) ||  /* AMR */
        !BestXCapabilityCheck(handle,BX_CAPABILITY_OBSERVER) )   /* OMR */
    {
      BX_TRY_RET=BX_E_OK;
    }
    else 
    {
      BX_TRY_FAIL(BX_E_NO_CAPABILITY);
    }  
    
    /* Range checking */
    BX_FCT_PARAM_CHK(1,transient>=BX_TRIGLINE_SIZE); 

    BX_TRY(BestXTrigTranDefaultSet(handle,transient));
    BX_TRY(BestXTrigCondDefaultSet(handle,transient));
  }

  BX_ERRETURN(BX_TRY_RET);
}


/*---------------------------------------------------------------------------*
 * BestXTrigDefaultSet()
 *
 * Purpose: Initialises the descriptor table of the trigger sequencer               
 *                                          
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXTrigDefaultSet(
  bx_handletype handle
)
{
  BX_DECLARE_FUNCNAME("BestXTrigDefaultSet []");

  bx_int32 t; /* transient */

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {    

    if( !BestXCapabilityCheck(handle,BX_CAPABILITY_ANALYZER) ||  /* AMR */
        !BestXCapabilityCheck(handle,BX_CAPABILITY_OBSERVER) )   /* OMR */
    {
      BX_TRY_RET=BX_E_OK;
    }
    else 
    {
      BX_TRY_FAIL(BX_E_NO_CAPABILITY);
    }  
    
    for (t=0;t<BX_TRIGLINE_SIZE;t++)
    {
      BX_TRY(BestXTrigTranCondDefaultSet(handle,t));
    }

    /* Here we have the same default line 128 times.
       To avoid programming all lines we reduce
       MaxNumLines to one.
    */

    assert(DB_TRIGSEQ.MaxNumLines==BX_TRIGLINE_SIZE); /* both one-based ! */
    DB_TRIGSEQ.MaxNumLines=1;

    /***************************************
       One state default trigger sequencer:
       Trigger, when BX_PATTERN_BUS0
       Store,   when BX_PATTERN_OBS0
    ***************************************/
    
    BX_TRY(BestXTrigTranSet(handle,0,BX_TRIGTRAN_STATE,0));
    BX_TRY(BestXTrigTranSet(handle,0,BX_TRIGTRAN_NEXTSTATE,0));
    BX_TRY(BestXTrigCondSet(handle,0,BX_TRIGCOND_X,"1"));
    BX_TRY(BestXTrigCondSet(handle,0,BX_TRIGCOND_TRIG,"bus0"));
    BX_TRY(BestXTrigCondSet(handle,0,BX_TRIGCOND_SQ,"obs0"));
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*****************************************************************************/
/* This call programs the  sequencer memory of the trigger counter       */
/* of the selected measure                                                   */
/*****************************************************************************/

bx_errtype EXPORT BestXTrigProg(
  bx_handletype handle
)
{
  BX_DECLARE_FUNCNAME("BestXTrigProg [trigprog]");

  bx_int32    val;

  BX_TRY_VARS_NO_PROG;
  
  BX_TRY_BEGIN
  {
    BX_LICENSECHECK(BX_CAPABILITY_ANALYZER|BX_CAPABILITY_CAPI); 
    
    /* We lock the resource analyzer */
    BX_TRY(BestXResourceLock(handle,BX_RESLOCK_ANALYZER));

    /* write the generic properties to card here */
    BX_TRY(BestXTrigGenGet(handle,BX_TRIGGEN_FBA,&val));
    BX_TRY(BestXDirectRegWrite(handle,BX_REG_TRIG_PL_A_REG,sizeof(bx_int32),val));
    BX_TRY(BestXTrigGenGet(handle,BX_TRIGGEN_FBB,&val));
    BX_TRY(BestXDirectRegWrite(handle,BX_REG_TRIG_PL_B_REG,sizeof(bx_int32),val));

    BX_TRY(BestXSequencerProg(handle,(bx_seqtype)BX_SEQ_TRIG));

    /* We unlock the resource analyzer */
    BX_TRY(BestXResourceUnlock(handle,BX_RESLOCK_ANALYZER));
  }

  BX_ERRETURN(BX_TRY_RET);
}


/*---------------------------------------------------------------------------*
 * BestXTrigMaxNumLinesGet()
 *
 * internal GUI helper function, do not dokument                                         
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXTrigMaxNumLinesGet(
  bx_handletype handle,
  bx_int32 *val
)
{
  BX_DECLARE_FUNCNAME("BestXTrigMaxNumLinesGet []");
  BX_TRY_VARS_NO_PROG;
  
  assert(val);

  BX_TRY_BEGIN
  {
    *val=DB_TRIGSEQ.MaxNumLines;
  }  

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXTrigMaxNumLinesSet()
 *
 * internal GUI helper function, do not dokument                                         
 * use with care !!!
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXTrigMaxNumLinesSet(
  bx_handletype handle,
  bx_int32 val
)
{
  BX_DECLARE_FUNCNAME("BestXTrigMaxNumLinesSet []");
  BX_TRY_VARS_NO_PROG;
   
  BX_TRY_BEGIN
  {
    DB_TRIGSEQ.MaxNumLines=val;
  }  

  BX_ERRETURN(BX_TRY_RET);
}
